###############################################################################
#                                                                             #
# Copyright (c) 2020 Plotly, Inc                                              #
#                                                                             #
# Embeddable for Dash Embedded is only licensed for use with the express      #
# written permission of Plotly, Inc.                                          #
#                                                                             #
# The above copyright notice and this permission notice shall be included in  #
# all copies or substantial portions of the Software.                         #
#                                                                             #
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR  #
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,    #
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE #
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER      #
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING     #
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER         #
# DEALINGS IN THE SOFTWARE.                                                   #
#                                                                             #
###############################################################################


import functools
import traceback

import os
import jwt
import flask


class EmbeddedAuth:
    """
    Embedded auth wrapper, all apps share the same secret key.
    """

    def __init__(self, apps, secret_key, claims, algorithm="HS512"):
        """
        :param apps: Apps to wraps.
        :param secret_key: The key to secure the user auth token with.
        :param claims: The claims to verify and encode the tokens.
        Note that the standard  `aud` claim needs to be specified as the `audience` key.
        For example: {'audience': <your-audience-claim>, 'iss': 'Dash Embedded'}
        :param algorithm: The algorithm to use for decoding/encoding.
        """
        self.apps = apps
        self.secret_key = secret_key
        self.claims = claims
        self.algorithm = algorithm

        self._protect_routes()

    # Note: `decode_token` is part of the public facing API
    # that users can override via subclassing
    def decode_token(self, token):
        is_request_from_snapshots = flask.request.headers.get('Plotly-User-Agent') == 'SNAPSHOT-ENGINE'
        secret = os.getenv("DASH_SECRET_KEY") if is_request_from_snapshots else self.secret_key
        algorithm = "HS256" if is_request_from_snapshots else self.algorithm

        try:
            return jwt.decode(
                token,
                secret,
                algorithms=[algorithm],
                **self.claims
            )
        except (jwt.InvalidTokenError, jwt.InvalidAudienceError, jwt.ExpiredSignatureError):
            print('Decoding token "{}" failed using {} algorithm. Headers & traceback below:'.format(
                token, algorithm
            ))
            print(flask.request.headers)
            traceback.print_exc()
            flask.abort(403)

    def create_token(self, info):
        payload = info.copy()
        payload.update(self.claims)
        return jwt.encode(payload, self.secret_key, algorithm=self.algorithm)

    def _protect_routes(self):
        for app in self.apps:
            for name, method in (
                    (x, app.server.view_functions[x])
                    for x in app.routes
                    if x in app.server.view_functions and "_dash-component-suites" not in x
            ):
                app.server.view_functions[name] = self.protect_route(method)

    def protect_route(self, func):
        @functools.wraps(func)
        def wrap(*args, **kwargs):

            # Do not protect CORS routes
            # TODO is this a security problem?
            if flask.request.method == 'OPTIONS':
                return func(*args, **kwargs)

            # Since some of the route are xhr application/json,
            # we can't get the normal authorization header,
            # it will be prefixed with HTTP_ instead.
            token = flask.request.authorization or flask.request.headers.environ.get(
                "HTTP_AUTHORIZATION"
            )
            if not token:
                print('Authorization token not found in request. Headers below:')
                print(flask.request.headers)
                flask.abort(403)
            self.decode_token(token.replace("Bearer ", ""))
            return func(*args, **kwargs)

        return wrap

    def get_user_data(self):
        token = flask.request.authorization or flask.request.headers.environ.get(
            "HTTP_AUTHORIZATION"
        )
        data = self.decode_token(token.replace("Bearer ", ""))
        return data
