###############################################################################
#                                                                             #
# Copyright (c) 2020 Plotly, Inc                                              #
#                                                                             #
# Embeddable for Dash Embedded is only licensed for use with the express      #
# written permission of Plotly, Inc.                                          #
#                                                                             #
# The above copyright notice and this permission notice shall be included in  #
# all copies or substantial portions of the Software.                         #
#                                                                             #
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR  #
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,    #
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE #
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER      #
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING     #
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER         #
# DEALINGS IN THE SOFTWARE.                                                   #
#                                                                             #
###############################################################################

from __future__ import print_function
import traceback
import flask
import dash as _dash
from pkg_resources import parse_version
from flask_cors import CORS


def get_resource_path(prefix):
    if parse_version(_dash.__version__) < parse_version('1.11.0'):
        return '{}_resources'.format(prefix)
    return '_resources'


class Embeddable:  # pylint: disable=R0903
    """
    Plugin to allow a dash app to be embedded in another web page

    Args are passed along to `flask_cors.CORS` to allow access from the
    enclosing app on a different domain. By default CORS is not enabled,
    but if you provide either `origins` or some other kwargs such as
    `resources`, CORS will be invoked with these args.
    """
    def __init__(self, origins=None, supports_credentials=True, **kwargs):
        self.origins = origins
        self.supports_credentials = supports_credentials
        self.cors_kwargs = kwargs

    def plug(self, app):
        self.app = app  # pylint: disable=W0201
        prefix = app.config['routes_pathname_prefix']
        app._add_url(get_resource_path(prefix), self._route_resources)  # pylint: disable=W0212
        app._embedded = True  # pylint: disable=W0212

        if self.origins or self.cors_kwargs:
            CORS(app.server,
                 supports_credentials=self.supports_credentials,
                 origins=self.origins,
                 **self.cors_kwargs)

        @app.server.errorhandler(Exception)
        def _wrap_errors(error):
            traceback.print_exc()
            try:
                description = error.description
                code = error.code
                return description, code
            except:  # pylint: disable=W0702
                return "The server encountered an error, please check the application's logs.", 500

    def _route_resources(self):
        app = self.app
        prefix = app.config['requests_pathname_prefix']

        def clean_one(resource, attr):
            resource_str = (resource.get(attr) if isinstance(resource, dict)
                            else resource)
            if resource_str.startswith(prefix):
                return '/' + resource_str[len(prefix):]
            return resource_str

        def clean_resources(resources, attr):
            return [clean_one(resource, attr) for resource in resources]

        js_resources = app._collect_and_register_resources(  # pylint: disable=W0212
            app.scripts.get_all_scripts(
                dev_bundles=app._dev_tools.serve_dev_bundles  # pylint: disable=W0212
            )
        )
        css_resources = app._collect_and_register_resources(  # pylint: disable=W0212
            app.css.get_all_css()
        )

        return flask.jsonify({
            'js': clean_resources(js_resources, 'src'),
            'css': clean_resources(css_resources, 'href'),
            'config': app._config()  # pylint: disable=W0212
        })
